<?php
// app/Controllers/ProductController.php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Models\ProductRepository;
use App\Models\SupplierRepository;

class ProductController extends Controller
{
    private ProductRepository $repo;
    private SupplierRepository $supplierRepo;

    public function __construct(Database $db)
    {
        parent::__construct();
        $this->repo = new ProductRepository($db);
        $this->supplierRepo = new SupplierRepository($db);
    }

    // seznam produktů
    public function index()
    {
        // vrací pole ProductEntity
        $items = $this->repo->findAllWithSupplier();
        $this->view->render('products/index', ['products' => $items, "title" => "Všechny produkty"]);
    }

    // ovoce
    public function ovoce()
    {
        $items = $this->repo->findByCategory("ovoce");
        $this->view->render('products/index', ['products' => $items, "title" => "Ovoce"]);
    }

    // zelenina
    public function zelenina()
    {
        $items = $this->repo->findByCategory("zelenina");
        $this->view->render('products/index', ['products' => $items, "title" => "Zelenina"]);
    }

    // orechy
    public function orechy()
    {
        $items = $this->repo->findByCategory("orechy");
        $this->view->render('products/index', ['products' => $items, "title" => "Ořechy"]);
    }

    // přidání produktu
    public function create()
    {
        // suppliers jsou SupplierEntity objekty
        $suppliers = $this->supplierRepo->findAll();
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = $this->validate($_POST);
            $this->repo->create($data); // create přijímá pole (forma)
            $this->redirect('?controller=product&action=index');
        }
        $this->view->render('products/form', ['suppliers' => $suppliers]);
    }

    // editace produktu
    public function edit()
    {
        $id = (int)($_GET['id'] ?? 0);
        /** @var \App\Models\Entities\ProductEntity|null $item */
        $item = $this->repo->find($id);
        if (!$item) {
            http_response_code(404);
            echo 'Produkt nenalezen';
            exit;
        }
        $suppliers = $this->supplierRepo->findAll();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = $this->validate($_POST);
            $this->repo->update($id, $data);
            $this->redirect('?controller=product&action=index');
        }

        $this->view->render('products/edit', ['product' => $item, 'suppliers' => $suppliers]);
    }

    // smazání produktu
    public function delete()
    {
        $id = (int)($_GET['id'] ?? 0);
        $this->repo->delete($id);
        $this->redirect('?controller=product&action=index');
    }

    // zobrazení detailu produktu
    public function show()
    {
        $id = (int)($_GET['id'] ?? 0);
        /** @var \App\Models\Entities\ProductEntity|null $item */
        $item = $this->repo->find($id);
        if (!$item) {
            http_response_code(404);
            echo 'Produkt nenalezen';
            exit;
        }
        // pokud chceme i jméno dodavatele, můžeme dohledat přes supplierRepo (vrací SupplierEntity)
        $supplier = null;
        if (!empty($item->supplier_id)) {
            $supplier = $this->supplierRepo->find((int)$item->supplier_id);
        }
        $this->view->render('products/show', ['product' => $item, 'supplier' => $supplier]);
    }

    // jednoduchá validace a převod typů (vrací pole vhodné pro create/update)
    private function validate(array $input): array
    {
        return [
            'name' => trim($input['name'] ?? ''),
            'category' => in_array($input['category'] ?? '', ['ovoce','zelenina','orechy'], true) ? $input['category'] : 'ovoce',
            'price' => isset($input['price']) ? (float)$input['price'] : 0.0,
            'stock' => isset($input['stock']) ? (int)$input['stock'] : 0,
            'supplier_id' => !empty($input['supplier_id']) ? (int)$input['supplier_id'] : null
        ];
    }
}